const { Sequelize } = require('sequelize');

const db = require('../../../../config/database.config');

const errorResponse = require('../../../../utils/errorResponse');

const successResponse = require('../../../../utils/successResponse');

const CampeignsModel = db.model.campeigns;
exports.CampeignsModel = CampeignsModel;
const campeignCategoryModel = db.model.campeignCategory;

exports.create = async (req, res) => {
    try {
        const {
            title,
            campeign_cat,
            campeign_date,
            short_description,
            long_description,
            status,
            target_amount,
        } = req.body;
        const { files } = req;

        const campeign_img = files?.find((file) => file?.fieldname === 'campeign_img')?.filename;

        const campeignData = {
            title,
            campeign_cat,
            campeign_date,
            campeign_img,
            short_description,
            long_description,
            status,
            target_amount,
        };

        const result = await CampeignsModel.create(campeignData);

        if (!result) {
            return errorResponse(404, 'FAILED', 'service created failed', res);
        }

        return successResponse(201, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// get all

// Fetch the campeign categories matching the IDs in the array

async function findcampeignCatData(arrIds) {
    try {
        const campeignCategory = await campeignCategoryModel.findAll({
            where: {
                id: {
                    [Sequelize.Op.in]: arrIds,
                },
            },

            attributes: ['id', 'title'],
        });

        return campeignCategory;
    } catch (err) {
        console.log(err);
        return [];
    }
}
exports.findcampeignCatData = findcampeignCatData;

exports.getAll = async (_, res) => {
    try {
        const result = await CampeignsModel.findAll();

        const campeignCatArr = result.reduce((acc, cur) => {
            const catArr = cur.campeign_cat ? cur.campeign_cat.split(',') : [];
            catArr.forEach((catId) => {
                if (!acc.includes(catId)) {
                    acc.push(catId);
                }
            });
            return acc;
        }, []);

        const campeignCategory = await findcampeignCatData(campeignCatArr);

        const dataWithCat = result.map((item) => {
            const catIds = item.campeign_cat ? item.campeign_cat.split(',') : [];
            const catArrData = catIds.map((catId) =>
                campeignCategory.find((cat) => cat.id === Number(catId))
            );
            return {
                ...item.dataValues,
                campeign_category: catArrData,
            };
        });

        return successResponse(200, 'OK', dataWithCat, res);
    } catch (error) {
        return errorResponse(500, 'ERROR', error.message || 'There is a server side problem', res);
    }
};

// get one

exports.getOne = async (req, res) => {
    try {
        const { id } = req.params;
        const { status } = req.query;

        const whereCondition = {
            id,
        };

        if (status === 'active') {
            whereCondition.status = 1;
        }

        // Fetch the campeign entry
        const result = await CampeignsModel.findOne({
            where: whereCondition,
        });

        // If the campeign is not found, return a 404 response
        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
        }

        // Split the campeign_cat string into an array
        const campeignCatArr = result.campeign_cat ? result.campeign_cat.split(',') : [];

        // Fetch the campeign categories matching the IDs in the array
        const campeignCategory = await campeignCategoryModel.findAll({
            where: {
                id: {
                    [Sequelize.Op.in]: campeignCatArr,
                },
            },

            attributes: ['id', 'title'],
        });

        // Add the retrieved categories to the campeign result
        result.dataValues.campeign_category = campeignCategory;

        result.dataValues.donation_collected = 100;

        // Return the success response with the campeign and its categories
        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

// update

exports.update = async (req, res) => {
    try {
        const { id } = req.params;
        const {
            title,
            campeign_cat,
            campeign_date,
            short_description,
            long_description,
            status,
            target_amount,
        } = req.body;

        console.log(req.body);

        const { files } = req;

        const isExist = await CampeignsModel.findOne({
            where: {
                id,
            },
        });

        if (!isExist) {
            return errorResponse(404, 'NOT FOUND', 'No Data found with this id', res);
        }

        const campeign_img =
            files?.find((file) => file?.fieldname === 'campeign_img')?.filename ||
            isExist?.campeign_img;

        const campeignData = {
            title,
            campeign_cat,
            campeign_date,
            campeign_img,
            short_description,
            long_description,
            status,
            target_amount,
        };
        const result = await CampeignsModel.update(campeignData, {
            where: {
                id,
            },
        });

        if (!result.includes(1)) {
            return errorResponse(404, 'FAILED', 'Campeigns updated failed', res);
        }

        return successResponse(201, 'OK', 'Update Successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// delete

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await CampeignsModel.destroy({
            where: {
                id,
            },
        });

        if (result !== 1) {
            return errorResponse(404, 'FAILED', 'Hero section modal deleted failed', res);
        }

        return successResponse(204, 'OK', 'Hero section deleted successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', 'There is a server side error', res);
    }
};
