const { Op } = require('sequelize');
const errorResponse = require('../../../../utils/errorResponse');
const successResponse = require('../../../../utils/successResponse');
const { BlogsModel, findBlogCatData } = require('./blogs.controller');

/**
 *
 * @param cat_id -- optional -- if cat id passed it will filter on that
 *
 * @returns -- Blogs[] -- all active blogs Data
 * * */

const getAllActiveBlogData = async (cat_id) => {
    try {
        const whereCondition = {
            status: 1,
        };
        console.log(cat_id);

        if (cat_id) {
            whereCondition.blog_cat = {
                [Op.like]: `%${cat_id}%`,
            };
        }

        const result = await BlogsModel.findAll({
            where: whereCondition,
        });
        console.log(result);
        return result;
    } catch (err) {
        console.log(err);
        return null;
    }
};

/**
 * @param  limit -- required -- if limit passed it will limit the result
 * @param page -- required -- if page passed it will filter on that
 *
 * @param cat_id : number -- optional -- if cat id passed it will filter on that
 *
 * @returns -- Blogs[] -- all active blogs Data
 * * */

const getBlogDataWithLimit = async (limit = 2, page = 1, cat_id) => {
  try {
    const offset = limit * (page - 1);

    const whereCondition = {
      status: 1,
    };

    if (cat_id) {
      whereCondition.blog_cat = {
        [Op.like]: `%${cat_id}%`,
      };
    }

    const result = await BlogsModel.findAll({
      where: whereCondition,
      limit: Number(limit),
      offset: Number(offset),
    });

    const blogCatArr = result.reduce((acc, cur) => {
      const catArr = cur.blog_cat ? cur.blog_cat.split(',') : [];
      catArr.forEach((catId) => {
        if (!acc.includes(catId)) {
          acc.push(catId);
        }
      });
      return acc;
    }, []);

    const blogCategory = await findBlogCatData(blogCatArr);

    const dataWithCat = result.map((item) => {
      const catIds = item.blog_cat ? item.blog_cat.split(',') : [];
      const catArrData = catIds.map((catId) =>
        blogCategory.find((cat) => cat.id === Number(catId)),
      );
      return {
        ...item.dataValues,
        blog_category: catArrData,
      };
    });

    const totalItems = await BlogsModel.count({
      where: {
        status: 1,
      },
    });
    const totalPages = Math.ceil(totalItems / limit);

    hasNextPage = page < totalPages;

    return {
      data: dataWithCat,
      metaData: {
        hasNextPage,
        totalPages,
      },
    };
  } catch (err) {
    console.log(err);
  }
};

getActiveData = async (req, res) => {
  try {
    const { limit, page } = req.query;

    let result = null;

    if (limit && page) {
      result = await getBlogDataWithLimit(limit, page);
    } else {
      result = await getAllActiveBlogData();
    }

    // if (!result) {
    //   return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
    // }
    //
    return successResponse(200, 'OK', result, res);
  } catch (err) {
    return errorResponse(
      500,
      'ERROR',
      err.message || 'There was a server side Error',
      res,
    );
  }
};

module.exports = {
  getActiveData,
  getBlogDataWithLimit,
  getAllActiveBlogData,
};
