const { Sequelize, Op } = require("sequelize");

const db = require("../../../../config/database.config");

const errorResponse = require("../../../../utils/errorResponse");

const successResponse = require("../../../../utils/successResponse");

const BlogCatModel = db.model.blogCategory;

const BlogModel = db.model.blogs;

exports.create = async (req, res) => {
  try {
    const { title, info, status } = req.body;

    const blogCat = {
      title,
      info,
      status,
    };
    const result = await BlogCatModel.create(blogCat);

    if (!result) {
      return errorResponse(404, "FAILED", "blog category created failed", res);
    }

    return successResponse(201, "OK", result, res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err?.message || "There is a server side problem",
      res,
    );
  }
};

// get all

exports.getAll = async (_, res) => {
  try {
    const result = await BlogCatModel.findAll();

    return successResponse(200, "OK", result, res);
  } catch (error) {
    return errorResponse(
      500,
      "ERROR",
      error.message || "There is a server side problem",
      res,
    );
  }
};

// get one

exports.getOne = async (req, res) => {
  try {
    const { id } = req.params;
    const result = await BlogCatModel.findOne({
      where: {
        id,
      },
    });

    if (!result) {
      return errorResponse(404, "NOT FOUND", "NO Active data found", res);
    }

    return successResponse(200, "OK", result, res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err.message || "There was a server side Error",
      res,
    );
  }
};

async function findBlogsByCategory() {
  const categories = await BlogCatModel.findAll({
    include: {
      model: BlogModel,
      as: "blog_list",
      attributes: [], // Avoid selecting blog columns (optional)
    },
  });

  const blogsByCategory = {};
  for (const category of categories) {
    blogsByCategory[category.title] = {
      name: category.name,
      blogs: await BlogModel.findAll({
        where: {
          [Op.fn]: [
            Sequelize.fn("split", Sequelize.col("blog_cat"), ","),
            "includes",
            category.id,
          ],
        },
      }),
    };
  }
  return blogsByCategory;
}

exports.getActiveData = async (_, res) => {
  try {
    const result = await BlogCatModel.findAll({
      where: {
        status: 1,
      },
    });

    const data = await Promise.all(
      result.map(async (category) => {
        const blogs = await BlogModel.findAll({
          where: {
            blog_cat: {
              [Op.like]: `%${category.id}%`,
            },
          },
        });

        return {
          ...category.dataValues,
          blog_list: blogs,
        };
      }),
    );

    if (!result) {
      return errorResponse(404, "NOT FOUND", "NO Active data found", res);
    }

    return successResponse(200, "OK", data, res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err.message || "There was a server side Error",
      res,
    );
  }
};

// update

exports.update = async (req, res) => {
  try {
    const { id } = req.params;
    const { title, info, status } = req.body;
    const { files } = req;

    const isExist = await BlogCatModel.findOne({
      where: {
        id,
      },
    });

    if (!isExist) {
      return errorResponse(404, "NOT FOUND", "No Data found with this id", res);
    }

    const blogData = {
      title,
      info,
      status,
    };
    const result = await BlogCatModel.update(blogData, {
      where: {
        id,
      },
    });

    if (!result.includes(1)) {
      return errorResponse(404, "FAILED", "Blogs Data created failed", res);
    }

    return successResponse(201, "OK", "Update Successfully", res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err?.message || "There is a server side problem",
      res,
    );
  }
};

// delete

exports.delete = async (req, res) => {
  try {
    const { id } = req.params;

    const result = await BlogCatModel.destroy({
      where: {
        id,
      },
    });

    if (result !== 1) {
      return errorResponse(404, "FAILED", "Blog Cat modal deleted failed", res);
    }

    return successResponse(204, "OK", "Blog Cat deleted successfully", res);
  } catch (err) {
    return errorResponse(500, "ERROR", "There is a server side error", res);
  }
};
