const { Op } = require('sequelize');
const db = require('../../../../config/database.config');
const errorResponse = require('../../../../utils/errorResponse');
const successResponse = require('../../../../utils/successResponse');

// dbs

const MenuGroup = db.model.menuGroup;
const MenuItemModel = db.model.menuItems;

// functions

const formatString = (inputString) => {
    if (inputString === null || inputString === undefined) {
        return '';
    }

    // Convert inputString to lowercase and replace spaces with hyphens
    return inputString.toString().replace(/\s+/g, '_').toLowerCase();
};

exports.create = async (req, res) => {
    try {
        const reqData = req.body;
        console.log(reqData);
        const { files } = req;

        const logo = files?.find((file) => file?.fieldname === 'logo')?.filename;

        const { menu_name, menu_for, status } = reqData;

        if (!menu_name || !menu_for) {
            return errorResponse(403, 'INVALID', 'Invalid menu input', res);
        }

        const storingObject = {
            menu_name,
            menu_for,
            menu_url: formatString(menu_name),
            logo: logo || null,
            status,
        };

        const results = await MenuGroup.create(storingObject);

        if (!results) {
            return errorResponse(404, 'FAILED', 'Failed to create new menu Group', res);
        }

        return successResponse(201, 'OK', results, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error creating the menu group',
            res
        );
    }
};

// find all the menu groups

exports.findAll = async (req, res) => {
    try {
        const results = await MenuGroup.findAll();

        return successResponse(201, 'OK', results, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error finding the menu group',
            res
        );
    }
};

// find all active groups menu

exports.findActive = async (req, res) => {
    try {
        const results = await MenuGroup.findAll({
            where: {
                status: 1,
            },
        });

        return successResponse(201, 'OK', results, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error finding active the menu group',
            res
        );
    }
};

exports.findActiveMenuForWeb = async (req, res) => {
    try {
        const results = await MenuGroup.findOne({
            where: {
                status: 1,
                menu_for: 2,
            },
        });

        const menuData = await MenuItemModel.findAll({
            where: {
                parent_menu_id: { [Op.or]: [0, null] },
            },
            include: [
                {
                    model: MenuItemModel,
                    as: 'subMenuItems',
                    order: [['menu_order', 'ASC']],
                    separate: true,
                },
            ],
            order: [['menu_order', 'ASC']],
        });

        const menuWithGroup = {
            ...results.dataValues,
            menu_items: menuData,
        };

        return successResponse(201, 'OK', menuWithGroup, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error finding active the menu group',
            res
        );
    }
};

exports.update = async (req, res) => {
    try {
        const { id } = req.params;
        const reqData = req.body;

        const { files } = req;

        const logo = files?.find((file) => file?.fieldname === 'logo')?.filename;

        const isExists = await MenuGroup.findOne({
            where: {
                id,
            },
        });

        if (!isExists) {
            return errorResponse(404, 'NOT FOUND', 'NO menu group found with this id', res);
        }

        const { menu_name, menu_for, status } = reqData;

        const storingObject = {
            menu_name,
            menu_for,
            // menu_url: formatString(menu_name),
            logo: logo || isExists?.logo,
            status,
        };

        const results = await MenuGroup.update(storingObject, {
            where: {
                id,
            },
        });

        if (!results.includes(1)) {
            return errorResponse(404, 'FAILED', 'Failed to updating new menu Group', res);
        }

        return successResponse(201, 'OK', 'Menu Group updated Successfully', res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error updating the menu group',
            res
        );
    }
};

// delete single menu group

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const results = await MenuGroup.destroy({
            where: {
                id,
            },
        });

        if (results !== 1) {
            return errorResponse(404, 'FAILED', 'Deleting menu group failed', res);
        }

        return successResponse(201, 'OK', results, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error finding the menu group',
            res
        );
    }
};
