const db = require('../../../../config/database.config');

const errorResponse = require('../../../../utils/errorResponse');

const successResponse = require('../../../../utils/successResponse');

// database

const FooterModel = db.model.footer;
const SocialIconListModel = db.model.social_icon_list;
const QuickLinkModel = db.model.quick_link;
const BottomBarModel = db.model.footer_bottom_bar;

exports.imageUpload = async (req, res) => {
    try {
        const files = req?.files;
        console.log(files);
        if (!files) {
            return errorResponse(404, 'FAILED', 'No file found', res);
        }

        return successResponse(200, 'OK', files, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side error', res);
    }
};

exports.create = async (req, res) => {
    try {
        const {
            description,
            logo,
            email,
            mobile,
            openhours,
            address,
            contact_info,
            status,
            iconData,
            quickLinkData,
            copyright,
            bottomData,
            latlng,
        } = req.body;

        const footerData = {
            description,
            contact_info,
            email,
            mobile,
            openhours,
            address,
            logo,
            copyright,
            status,
            latlng,
        };

        const result = await FooterModel.create(footerData);

        if (!result) {
            return errorResponse(404, 'FAILED', 'Failed to create new Footer', res);
        }

        const iconDataList =
            iconData?.length > 0 &&
            iconData.map((icon) => ({
                title: icon?.title,
                icon: icon?.icon,
                link: icon?.link,
                footer_id: result?.id,
                status: icon?.status,
            }));

        await SocialIconListModel.bulkCreate(iconDataList);

        const quickLinkList =
            quickLinkData?.length > 0 &&
            quickLinkData.map((link) => ({
                title: link?.title,
                link: link?.link,
                new_tab: link?.new_tab,
                footer_id: result?.id,
                status: link?.status,
            }));

        await QuickLinkModel.bulkCreate(quickLinkList);

        // bottom bar data

        const bottomBarData =
            bottomData?.length > 0 &&
            bottomData.map((icon) => ({
                title: icon?.title,
                link: icon?.link,
                footer_id: result?.id,
                status: icon?.status,
            }));

        await BottomBarModel.bulkCreate(bottomBarData);

        return successResponse(201, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

exports.getAll = async (_, res) => {
    try {
        const result = await FooterModel.findAll({
            include: [
                {
                    model: SocialIconListModel,
                },
                {
                    model: QuickLinkModel,
                },
                {
                    model: BottomBarModel,
                },
            ],
        });

        return successResponse(200, 'OK', result, res);
    } catch (error) {
        return errorResponse(500, 'ERROR', error.message || 'There is a server side problem', res);
    }
};

exports.getOne = async (req, res) => {
    try {
        const { id } = req.params;
        const result = await FooterModel.findOne({
            where: {
                id,
            },
            include: [
                {
                    model: SocialIconListModel,
                },
                {
                    model: QuickLinkModel,
                },
                {
                    model: BottomBarModel,
                },
            ],
        });

        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO footer data found', res);
        }

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

exports.getActiveData = async (_, res) => {
    try {
        const result = await FooterModel.findOne({
            where: {
                status: 1,
            },
            include: [
                {
                    model: SocialIconListModel,
                },
                {
                    model: QuickLinkModel,
                },
                {
                    model: BottomBarModel,
                },
            ],
        });

        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
        }

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

// update

exports.update = async (req, res) => {
    try {
        const { id } = req.params;

        const isExist = await FooterModel.findByPk(id);

        if (!isExist) {
            return errorResponse(404, 'NOT FOUND', 'No data found with this id', res);
        }

        const {
            description,
            email,
            mobile,
            openhours,
            logo,
            contact_info,
            status,
            iconData,
            quickLinkData,
            copyright,
            bottomData,
            latlng,
        } = req.body;

        const footerData = {
            description,
            contact_info,
            email,
            mobile,
            openhours,
            logo: logo || isExist?.logo,
            copyright,
            bottomData,
            status,
            latlng,
        };

        const result = await FooterModel.update(footerData, {
            where: {
                id,
            },
        });

        if (!result) {
            return errorResponse(404, 'FAILED', 'Failed to create new Footer', res);
        }

        const iconDataList =
            iconData?.length > 0 &&
            iconData.map((icon) => ({
                title: icon?.title,
                icon: icon?.icon,
                link: icon?.link,
                footer_id: id,
                status: icon?.status,
            }));
        if (iconDataList?.length > 0) {
            await SocialIconListModel.destroy({
                where: {
                    footer_id: id,
                },
            });
            await SocialIconListModel.bulkCreate(iconDataList);
        }

        const quickLinkList =
            quickLinkData?.length > 0 &&
            quickLinkData.map((link) => ({
                title: link?.title,
                link: link?.link,
                new_tab: link?.new_tab,
                footer_id: id,
                status: link?.status,
            }));

        if (quickLinkList?.length > 0) {
            await QuickLinkModel.destroy({
                where: {
                    footer_id: id,
                },
            });

            await QuickLinkModel.bulkCreate(quickLinkList);
        }

        // bottom bar data

        const bottomBarData =
            bottomData?.length > 0 &&
            bottomData.map((icon) => ({
                title: icon?.title,
                link: icon?.link,
                footer_id: id,
                status: icon?.status,
            }));

        if (bottomBarData?.length > 0) {
            await BottomBarModel.destroy({
                where: {
                    footer_id: id,
                },
            });

            await BottomBarModel.bulkCreate(bottomBarData);
        }

        if (!result.includes(1)) {
            return errorResponse(404, 'FAILED', 'Failed to update Footer', res);
        }

        return successResponse(201, 'OK', 'Footer Update Successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// delete heroSection

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await FooterModel.destroy({
            where: {
                id,
            },
        });

        if (result !== 1) {
            return errorResponse(404, 'FAILED', 'Hero section modal deleted failed', res);
        }

        // delete social icons
        await SocialIconListModel.destroy({
            where: {
                footer_id: id,
            },
        });

        // delete quickLink

        await QuickLinkModel.destroy({
            where: {
                footer_id: id,
            },
        });

        return successResponse(
            204,
            'OK',
            'Footer, social icon, quickLink deleted successfully',
            res
        );
    } catch (err) {
        return errorResponse(500, 'ERROR', 'There is a server side error', res);
    }
};
