const { Op } = require('sequelize');
const errorResponse = require('../../../../utils/errorResponse');
const successResponse = require('../../../../utils/successResponse');
const { EventsModel } = require('./events.controller');

/**
 *
 * @param cat_id -- optional -- if cat id passed it will filter on that
 *
 * @returns -- Events[] -- all active events Data
 * * */

const getAllActiveEventData = async (cat_id) => {
    try {
        const whereCondition = {
            status: 1,
        };
        console.log(cat_id);

        if (cat_id) {
            whereCondition.event_cat = {
                [Op.like]: `%${cat_id}%`,
            };
        }

        const result = await EventsModel.findAll({
            where: whereCondition,
        });
        console.log(result);
        return result;
    } catch (err) {
        console.log(err);
        return null;
    }
};

/**
 * @param  limit -- required -- if limit passed it will limit the result
 * @param page -- required -- if page passed it will filter on that
 *
 * @param cat_id : number -- optional -- if cat id passed it will filter on that
 *
 * @returns -- Events[] -- all active events Data
 * * */

const getEventDataWithLimit = async (limit = 2, page = 1, cat_id) => {
    try {
        const offset = limit * (page - 1);

        const whereCondition = {
            status: 1,
        };

        if (cat_id) {
            whereCondition.event_cat = {
                [Op.like]: `%${cat_id}%`,
            };
        }

        const result = await EventsModel.findAll({
            where: whereCondition,
            limit: Number(limit),
            offset: Number(offset),
        });

        const totalItems = await EventsModel.count({
            where: {
                status: 1,
            },
        });
        const totalPages = Math.ceil(totalItems / limit);

        const hasNextPage = page < totalPages;

        return {
            data: result,
            metaData: {
                hasNextPage,
                totalPages,
            },
        };
    } catch (err) {
        console.log(err);
        throw new Error(err);
    }
};

const getActiveEvent = async (req, res) => {
    try {
        const { limit, page } = req.query;

        let result = null;

        if (limit && page) {
            result = await getEventDataWithLimit(limit, page);
        } else {
            result = await getAllActiveEventData();
        }

        // if (!result) {
        //   return errorResponse(404, "NOT FOUND", "NO Active data found", res);
        // }
        //
        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

async function getLatestActiveEvents(req, res) {
    try {
        const result = await EventsModel.findOne({
            where: {
                status: 1,
                event_date: {
                    [Op.gte]: new Date(),
                },
            },
            order: [['event_date', 'ASC']], // Order by date in descending order
            limit: 1, // Limit to 1 result
        });

        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
        }

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
}

module.exports = {
    getActiveEvent,
    getEventDataWithLimit,
    getAllActiveEventData,
    getLatestActiveEvents,
};
