const { Sequelize, Op } = require('sequelize');

const db = require('../../../../config/database.config');

const errorResponse = require('../../../../utils/errorResponse');

const successResponse = require('../../../../utils/successResponse');

const CampeignCatModel = db.model.campeignCategory;

const CampeignModel = db.model.campeigns;

exports.create = async (req, res) => {
    try {
        const { title, info, status } = req.body;

        const campeignCat = {
            title,
            info,
            status,
        };
        const result = await CampeignCatModel.create(campeignCat);

        if (!result) {
            return errorResponse(404, 'FAILED', 'campeign category created failed', res);
        }

        return successResponse(201, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// get all

exports.getAll = async (_, res) => {
    try {
        const result = await CampeignCatModel.findAll();

        return successResponse(200, 'OK', result, res);
    } catch (error) {
        return errorResponse(500, 'ERROR', error.message || 'There is a server side problem', res);
    }
};

// get one

exports.getOne = async (req, res) => {
    try {
        const { id } = req.params;
        const result = await CampeignCatModel.findOne({
            where: {
                id,
            },
        });

        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
        }

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

async function findCampeignsByCategory() {
    const categories = await CampeignCatModel.findAll({
        include: {
            model: CampeignModel,
            as: 'campeign_list',
            attributes: [], // Avoid selecting campeign columns (optional)
        },
    });

    const campeignsByCategory = {};
    for (const category of categories) {
        campeignsByCategory[category.title] = {
            name: category.name,
            campeigns: await CampeignModel.findAll({
                where: {
                    [Op.fn]: [
                        Sequelize.fn('split', Sequelize.col('campeign_cat'), ','),
                        'includes',
                        category.id,
                    ],
                },
            }),
        };
    }
    return campeignsByCategory;
}

exports.getActiveData = async (_, res) => {
    try {
        const result = await CampeignCatModel.findAll({
            where: {
                status: 1,
            },
        });

        const data = await Promise.all(
            result.map(async (category) => {
                const campeigns = await CampeignModel.findAll({
                    where: {
                        campeign_cat: {
                            [Op.like]: `%${category.id}%`,
                        },
                    },
                });

                return {
                    ...category.dataValues,
                    campeign_list: campeigns,
                };
            })
        );

        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
        }

        return successResponse(200, 'OK', data, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

// update

exports.update = async (req, res) => {
    try {
        const { id } = req.params;
        const { title, info, status } = req.body;
        const { files } = req;

        const isExist = await CampeignCatModel.findOne({
            where: {
                id,
            },
        });

        if (!isExist) {
            return errorResponse(404, 'NOT FOUND', 'No Data found with this id', res);
        }

        const campeignData = {
            title,
            info,
            status,
        };
        const result = await CampeignCatModel.update(campeignData, {
            where: {
                id,
            },
        });

        if (!result.includes(1)) {
            return errorResponse(404, 'FAILED', 'Campeigns Data created failed', res);
        }

        return successResponse(201, 'OK', 'Update Successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// delete

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await CampeignCatModel.destroy({
            where: {
                id,
            },
        });

        if (result !== 1) {
            return errorResponse(404, 'FAILED', 'Campeign Cat modal deleted failed', res);
        }

        return successResponse(204, 'OK', 'Campeign Cat deleted successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', 'There is a server side error', res);
    }
};
