const { Sequelize } = require("sequelize");

const db = require("../../../../config/database.config");

const errorResponse = require("../../../../utils/errorResponse");

const successResponse = require("../../../../utils/successResponse");

const BlogsModel = db.model.blogs;
exports.BlogsModel = BlogsModel;
const BlogCategoryModel = db.model.blogCategory;

exports.create = async (req, res) => {
  try {
    const {
      title,
      blog_cat,
      blog_date,
      short_description,
      long_description,
      status,
    } = req.body;
    const { files } = req;

    const blog_img = files?.find(
      (file) => file?.fieldname === "blog_img",
    )?.filename;

    const blogData = {
      title,
      blog_cat,
      blog_date,
      blog_img,
      short_description,
      long_description,
      status,
    };

    const result = await BlogsModel.create(blogData);

    if (!result) {
      return errorResponse(404, "FAILED", "service created failed", res);
    }

    return successResponse(201, "OK", result, res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err?.message || "There is a server side problem",
      res,
    );
  }
};

// get all

// Fetch the blog categories matching the IDs in the array

async function findBlogCatData(arrIds) {
  try {
    const blogCategory = await BlogCategoryModel.findAll({
      where: {
        id: {
          [Sequelize.Op.in]: arrIds,
        },
      },

      attributes: ["id", "title"],
    });

    return blogCategory;
  } catch (err) {
    console.log(err);
    return [];
  }
}
exports.findBlogCatData = findBlogCatData;

exports.getAll = async (_, res) => {
  try {
    const result = await BlogsModel.findAll();

    const blogCatArr = result.reduce((acc, cur) => {
      const catArr = cur.blog_cat ? cur.blog_cat.split(",") : [];
      catArr.forEach((catId) => {
        if (!acc.includes(catId)) {
          acc.push(catId);
        }
      });
      return acc;
    }, []);

    const blogCategory = await findBlogCatData(blogCatArr);

    const dataWithCat = result.map((item) => {
      const catIds = item.blog_cat ? item.blog_cat.split(",") : [];
      const catArrData = catIds.map((catId) =>
        blogCategory.find((cat) => cat.id === Number(catId)),
      );
      return {
        ...item.dataValues,
        blog_category: catArrData,
      };
    });

    return successResponse(200, "OK", dataWithCat, res);
  } catch (error) {
    return errorResponse(
      500,
      "ERROR",
      error.message || "There is a server side problem",
      res,
    );
  }
};

// get one

exports.getOne = async (req, res) => {
  try {
    const { id } = req.params;
    const { status } = req.query;

    const whereCondition = {
      id,
    };

    if (status === "active") {
      whereCondition.status = 1;
    }

    // Fetch the blog entry
    const result = await BlogsModel.findOne({
      where: whereCondition,
    });

    // If the blog is not found, return a 404 response
    if (!result) {
      return errorResponse(404, "NOT FOUND", "NO Active data found", res);
    }

    // Split the blog_cat string into an array
    const blogCatArr = result.blog_cat ? result.blog_cat.split(",") : [];

    // Fetch the blog categories matching the IDs in the array
    const blogCategory = await BlogCategoryModel.findAll({
      where: {
        id: {
          [Sequelize.Op.in]: blogCatArr,
        },
      },

      attributes: ["id", "title"],
    });

    // Add the retrieved categories to the blog result
    result.dataValues.blog_category = blogCategory;

    // Return the success response with the blog and its categories
    return successResponse(200, "OK", result, res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err.message || "There was a server side Error",
      res,
    );
  }
};

// update

exports.update = async (req, res) => {
  try {
    const { id } = req.params;
    const {
      title,
      blog_cat,
      blog_date,
      short_description,
      long_description,
      status,
    } = req.body;

    console.log(req.body);

    const { files } = req;

    const isExist = await BlogsModel.findOne({
      where: {
        id,
      },
    });

    if (!isExist) {
      return errorResponse(404, "NOT FOUND", "No Data found with this id", res);
    }

    const blog_img =
      files?.find((file) => file?.fieldname === "blog_img")?.filename ||
      isExist?.blog_img;

    const blogData = {
      title,
      blog_cat,
      blog_date,
      blog_img,
      short_description,
      long_description,
      status,
    };
    const result = await BlogsModel.update(blogData, {
      where: {
        id,
      },
    });

    if (!result.includes(1)) {
      return errorResponse(404, "FAILED", "Blogs updated failed", res);
    }

    return successResponse(201, "OK", "Update Successfully", res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err?.message || "There is a server side problem",
      res,
    );
  }
};

// delete

exports.delete = async (req, res) => {
  try {
    const { id } = req.params;

    const result = await BlogsModel.destroy({
      where: {
        id,
      },
    });

    if (result !== 1) {
      return errorResponse(
        404,
        "FAILED",
        "Hero section modal deleted failed",
        res,
      );
    }

    return successResponse(204, "OK", "Hero section deleted successfully", res);
  } catch (err) {
    return errorResponse(500, "ERROR", "There is a server side error", res);
  }
};
