// logger.js
const morgan = require('morgan');
const chalk = require('chalk');

// Define custom tokens to colorize different parts of the log
morgan.token('methodColor', (req) => chalk.blue(req.method));
morgan.token('urlColor', (req) => chalk.magenta(req.url));
morgan.token('statusColor', (req, res) => {
    const status = res.statusCode;
    if (status >= 500) {
        return chalk.red(status);
    }
    if (status >= 400) {
        return chalk.yellow(status);
    }
    if (status >= 300) {
        return chalk.cyan(status);
    }
    if (status >= 200) {
        return chalk.green(status);
    }
    return status;
});
morgan.token('responseTimeColor', (req, res) => {
    const responseTime = res.responseTime || 0;
    return chalk.greenBright(`${responseTime} ms`);
});
morgan.token('contentLengthColor', (req, res) =>
    chalk.white(`${res.get('Content-Length') || '-'} bytes`)
);
morgan.token('ipColor', (req) => chalk.cyan(req.ip));
morgan.token('date', () => chalk.gray(new Date().toISOString()));

// Middleware to measure response time
const responseTimeMiddleware = (req, res, next) => {
    const start = Date.now();
    res.on('finish', () => {
        res.responseTime = Date.now() - start;
    });
    next();
};

// Custom Morgan format that includes the colored tokens
const format = (tokens, req, res) =>
    [
        tokens.ipColor(req, res),
        '- -',
        `[${tokens.date(req, res)}]`,
        `"${tokens.methodColor(req, res)} ${tokens.urlColor(req, res)} HTTP/${req.httpVersion}"`,
        tokens.statusColor(req, res),
        tokens.contentLengthColor(req, res),
        `"${req.headers.referer || '-'}"`,
        `"${req.headers['user-agent'] || '-'}"`,
        tokens.responseTimeColor(req, res),
    ].join(' ');

// Export the configured morgan middleware and responseTimeMiddleware
const logger = morgan(format);
module.exports = { logger, responseTimeMiddleware };
