const { Sequelize, Op } = require('sequelize');

const db = require('../../../../config/database.config');

const errorResponse = require('../../../../utils/errorResponse');

const successResponse = require('../../../../utils/successResponse');

const CourseCatModel = db.model.courseCategory;

const CourseModel = db.model.courses;

exports.create = async (req, res) => {
    try {
        const { title, info, status } = req.body;

        const courseCat = {
            title,
            info,
            status,
        };
        const result = await CourseCatModel.create(courseCat);

        if (!result) {
            return errorResponse(404, 'FAILED', 'course category created failed', res);
        }

        return successResponse(201, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// get all

exports.getAll = async (_, res) => {
    try {
        const result = await CourseCatModel.findAll();

        return successResponse(200, 'OK', result, res);
    } catch (error) {
        return errorResponse(500, 'ERROR', error.message || 'There is a server side problem', res);
    }
};

// get one

exports.getOne = async (req, res) => {
    try {
        const { id } = req.params;
        const result = await CourseCatModel.findOne({
            where: {
                id,
            },
        });

        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
        }

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

async function findCoursesByCategory() {
    const categories = await CourseCatModel.findAll({
        include: {
            model: CourseModel,
            as: 'course_list',
            attributes: [], // Avoid selecting course columns (optional)
        },
    });

    const coursesByCategory = {};
    for (const category of categories) {
        coursesByCategory[category.title] = {
            name: category.name,
            courses: await CourseModel.findAll({
                where: {
                    [Op.fn]: [
                        Sequelize.fn('split', Sequelize.col('course_cat'), ','),
                        'includes',
                        category.id,
                    ],
                },
            }),
        };
    }
    return coursesByCategory;
}

exports.getActiveData = async (_, res) => {
    try {
        const result = await CourseCatModel.findAll({
            where: {
                status: 1,
            },
        });

        const data = await Promise.all(
            result.map(async (category) => {
                const courses = await CourseModel.findAll({
                    where: {
                        course_cat: {
                            [Op.like]: `%${category.id}%`,
                        },
                    },
                });

                return {
                    ...category.dataValues,
                    course_list: courses,
                };
            })
        );

        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
        }

        return successResponse(200, 'OK', data, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

// update

exports.update = async (req, res) => {
    try {
        const { id } = req.params;
        const { title, info, status } = req.body;
        const { files } = req;

        const isExist = await CourseCatModel.findOne({
            where: {
                id,
            },
        });

        if (!isExist) {
            return errorResponse(404, 'NOT FOUND', 'No Data found with this id', res);
        }

        const courseData = {
            title,
            info,
            status,
        };
        const result = await CourseCatModel.update(courseData, {
            where: {
                id,
            },
        });

        if (!result.includes(1)) {
            return errorResponse(404, 'FAILED', 'Courses Data created failed', res);
        }

        return successResponse(201, 'OK', 'Update Successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// delete

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await CourseCatModel.destroy({
            where: {
                id,
            },
        });

        if (result !== 1) {
            return errorResponse(404, 'FAILED', 'Course Cat modal deleted failed', res);
        }

        return successResponse(204, 'OK', 'Course Cat deleted successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', 'There is a server side error', res);
    }
};
